"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteCategoria = exports.getOneCategoria = exports.getAllCategorias = exports.createCategoria = void 0;
const express_validator_1 = require("express-validator");
const mongoose_1 = __importDefault(require("mongoose"));
const categoria_1 = __importDefault(require("../models/categoria"));
const createCategoria = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { nome, pai, descricao } = req.body;
        // Verificar se a categoria já existe
        const verifyCategoria = await categoria_1.default.findOne({ nome });
        if (verifyCategoria) {
            const error = new Error(`A Categoria ${verifyCategoria.nome} já foi criada!`);
            error.statusCode = 422;
            throw error;
        }
        // Criar nova categoria
        const categoria = new categoria_1.default({
            nome,
            descricao,
            pai: pai || null // Define `pai` como null se não houver pai
        });
        // Se houver um pai, verificar se ele existe
        if (pai) {
            // Verificar se o `pai` é um ObjectId válido
            if (!mongoose_1.default.Types.ObjectId.isValid(pai)) {
                const error = new Error('O ID da Categoria pai é inválido!');
                error.statusCode = 422;
                throw error;
            }
            // Verificar se a Categoria pai existe
            const verifyPaiCategoria = await categoria_1.default.findById(pai);
            if (!verifyPaiCategoria) {
                const error = new Error('A Categoria pai não existe!');
                error.statusCode = 422;
                throw error;
            }
            // Adicionar a subcategoria à Categoria pai
            verifyPaiCategoria.subcategorias.push(categoria._id);
            await categoria.save();
            await verifyPaiCategoria.save();
        }
        else {
            await categoria.save();
        }
        res.status(201).json({
            msg: `${pai ? 'Subcategoria' : 'Categoria'} criada com sucesso`,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createCategoria = createCategoria;
const getAllCategorias = async (req, res, next) => {
    try {
        const categorias = await categoria_1.default.find({});
        if (categorias.length === 0) {
            return res.status(404).json({
                success: false,
                categorias
            });
        }
        res.status(200).json({
            success: true,
            categorias
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getAllCategorias = getAllCategorias;
const getOneCategoria = async (req, res, next) => {
    try {
        const id = req.params.id;
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (validId) {
            const categoria = await categoria_1.default.findOne({ _id: id });
            if (!categoria) {
                return res.status(404).json({
                    success: false,
                    categoria
                });
            }
            res.status(200).json({
                success: true,
                categoria
            });
        }
        else {
            return res.status(400).json({
                success: false,
                message: 'Id inválido.'
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getOneCategoria = getOneCategoria;
const deleteCategoria = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const id = req.params.id;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        // Busca o Categoria para garantir que ele existe
        const categoria = await categoria_1.default.findById(id);
        if (!categoria) {
            return res.status(404).json({
                success: false,
                message: 'Categoria não encontrada.',
            });
        }
        // Buscar e apagar todas as subcategorias
        await categoria_1.default.deleteMany({ pai: id });
        // Apagar a categoria principal
        await categoria_1.default.findByIdAndDelete(id);
        res.status(200).json({
            success: true,
            message: `Categoria ${categoria.nome} deletada com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.deleteCategoria = deleteCategoria;
